/*
   siemens.c

	Functions for Siemens 3-axis ISA-board
	
	Henry Palonen, h@yty.net, 12.11.2002
    
*/


/* ident tag */
#ifndef __GNUC__
#ifndef __attribute__
#define __attribute__(x)
#endif
#endif

// #include <linux/module.h>
#include <asm/io.h>

static char __attribute__((unused)) ident[] = "$Id: siemens.c,v 0.1 2002/11/12 20:57:00 henkka Exp $";


#if !defined(rtlinux) && !defined(rtai)
#include <stdio.h>
#endif

#include "siemens.h"               /* these decls */
#include "extintf.h"            /* EXT_ENCODER_INDEX_MODEL */

/* base address-- override default at compile time in siemens.h,
   or at module load time with STG_BASE_ADDRESS=value */
unsigned short STG_BASE_ADDRESS = DEFAULT_STG_BASE_ADDRESS;
int FIND_STG_BASE_ADDRESS=1;

void *card_address;

#if defined(rtlinux) || defined(rtai)
#ifndef MODULE
#define MODULE
#endif
#define DO_IT
#endif

#ifdef LINUX
#define DO_IT
/*
  Compiling this for LINUX (not RTLINUX) means linking this into a Linux
  process, running non-real-time in user space. This is useful for debugging.
  If this is done, then the following stuff needs to be done to access the
  IO space.
  */

/*
  Because of a limitation in gcc (present at least in 2.7.2.1 and below), you
  _have to_ compile any source code that uses these routines with optimisation
  turned on (gcc -O1 or higher), or alternatively #define extern to be
  empty before #including <asm/io.h>.
 */
#ifdef DEFINE_EXTERN_BEFORE_IO
#define extern
#endif


/*
  Need to access a 64 word block anywhere in the 64K IO space, for default
  SIEMENS base address. ioperm() only enables up to 0x3FF, so we need to use
  iopl() to grant full access to IO space.
  */
#include <unistd.h>             /* iopl() */

#endif /* LINUX  */

#ifdef DO_IT

// #if defined(__KERNEL__) && defined(linux)
#include <linux/kernel.h>
// #endif

#if defined(linux) || defined(rtlinux) || defined(rtai)
#include <sys/io.h>             /* ioperm() */
#endif

/* asm/io.h already included by sys/io.h for 2.2, 2.3, 2.4 kernels  */


#ifndef LINUX_VERSION_CODE
#include <linux/version.h>
#endif

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,2,0)
/* Linux or RT-Linux use io ports */
#include <asm/io.h>             /* outb, inb */
#endif

#else

/* otherwise do nothing */

#include "rcs_prnt.hh"

#endif
#define SIEMENS_INIT_WAIT 1000      /* usecs to wait after siemensMotInit() */

/*
  Here follows the routines to interface to the ISA Encoder DAC card.
   Most of the function descriptions have been lifted from extinf.h
  some of which will do nothing due to lack of IO.
*/


/* Init/quit functions */

/*
  siemensMotInit(const char * stuff)

  Call once before any of the other motion IO functions are called.
  'stuff' argument can be used to pass board-specific stuff like config
  files.
  */
int siemensMotInit(const char * stuff)
{
  int t;
#ifdef LINUX
  if (-1 == iopl(3))
    {
      return -1;
    }
#endif
//  printk("<1> INT_STATUS=%0x\n", t);
  //if ((INT_STATUS & 0x0030) != 0x0020 && (INT_STATUS & 0x0030) != 0x0010)
  //  {
/* Check for the presence of the card - If the ID bits are not correct,
   then the board is probably not here. */
  //     return -1;
  //  }
  /*
   Interrupts are not used by EMC (I think). So make sure they are
   masked - The card should power up at these settings, but still
   wise to force the defaults.
   */
//  LOGIC_CFG;			// Issue a reset to the board
//  INT_DIGCFG(0);		// Set all DIO to input and mask interrupts
//  ENC_RESET;			// Just one write to reset all encoder channels.
//  ENC_MASK03(0x0300);		// Mask axes 0-3 encoder interrupts and set Hi level
//  ENC_MASK47(0x0300);		// Mask axes 4-7 encoder interrupts and set Hi level
//  t = ENC_STAT03;		// Clear encoder status bits
//  t = ENC_STAT47;		// and also for axis 4 to 7

//  siemensDioInit(0);		// Configure the digital IO

  // STG_BASE_ADDRESS = ioremap(0x0d0000,0xffff);
  
  // remapping of card's io-address
  card_address = ioremap(0x0d0000, 0xffff);
  printk("<1>Siemens card address = %lx\n", card_address);
  // config bits
  outw(0x82d0, 0x300);
    
 for (t = 0; t < 2; t++)
   {
     RawDacOut(t, 0.0);		// Make sure DAC ouputs are at zero
    }
//  DAC_ENABLE;			// Enable the DAC outputs
   return 0;
}

/*
  siemensDioInit(const char * stuff)

  Call once before any of the other digital IO functions are called.
  'stuff' argument can be used to pass board-specific stuff like config
  files.
  */
int siemensDioInit(const char * stuff)
{
  INT_DIGCFG(DIO_CFG_MASK);
  siemensDioShortWrite(0, 0);
  return 0;
}

/*
  siemensDioQuit()

  Call once, after which point no other functions will be called until
  after a call to extDioInit().
  */
int siemensDioQuit(void)
{
  INT_DIGCFG(0);
  return 0;
}

/*
  siemensMotQuit()

  Call once, after which point no other functions will be called until
  after a call to siemensMotInit().
  */
int siemensMotQuit(void)
{
  DAC_DISARM;		/* disable the relay and set DAC outputs to 0V */
  siemensDioQuit();
  return 0;
}

/* DAC functions */

/*
  siemensDacNum()

  returns number of DACs in system which is 3 in this case.
  */
int siemensDacNum(void)
{
  return 3;
}

/*
  siemensDacWrite(int dac, double volts)

  writes the value to the DAC at indicated dac, 0 .. max DAC - 1.
  Value is in volts. Returns 0 if OK, -1 if dac is out of range.
  */
int siemensDacWrite(int dac, double volts)
{
  if ((dac > 7) || (dac < 0))
    {
      return -1;
    }
  //DAC_SETIND;			/* Set DACs to individual update mode */
  RawDacOut(dac, volts);
  return 0;
}

/*
  siemensDacWriteAll(int max, double * volts)

  writes the values to the DAC range. Values are array of DAC outputs
  in volts. Returns 0 if OK, -1 if not max is out of range. This is
  provided for systems in which DACs can be written all at once for
  speed or synchronicity. max is number of DACs, so for dacs 0..3
  max would be 4.
  */
int siemensDacWriteAll(int max, double * volts)
{
 int t;
  if ((max > 7) || (max < 0))
    {
      return -1;
    }
  //DAC_SETSIM;			// Set DAC to simultaneous update mode    
  for (t = 0; t < max; t++)
    {
      RawDacOut(t, volts[t]);
     }
  //DAC_UPDATE;			// Now update the DACs
  return 0;
}

/* Encoder functions */

/*
  siemensEncoderIndexModel()

  Returns the type of encoder indexing done by the board.

  The handling of index pulses for encoder homing is problematic. There
  are two models (at least) for how incremental encoders can be homed
  off the index pulse. The first assumes the encoder index pulse sets
  a flag when it occurs, but the count value doesn't change. The controller
  polls on this flag, and when it is seen to be latched the controller
  reads the position and uses this as an offset. There is a latency that
  makes this less accurate than the second method, in which the occurrence
  of the index pulse resets the count automatically. The problem is that
  the controller logic is different for the two methods, so if you replace
  a board which does it the first way with one that does it the second,
  you need to recode the logic. The function "siemensEncoderIndexModel()"
  returns the model used by the board, so at least you can detect it
  and if you have coded up both types you can switch automatically.

  EXT_ENCODER_INDEX_MODEL_MANUAL
  indicates that the index pulse sets a latch flag, but you have to
  read this and then the encoder value and handle offsets yourself.
  The board won't change its count value on the index pulse.

  EXT_ENCODER_INDEX_MODEL_AUTO
  indicates that the index pulse zeros the encoder count automatically.
  */

/* flags defined bit-exclusive for OR'ing if board can do multiple ways */
#define EXT_ENCODER_INDEX_MODEL_MANUAL 0x00000001
#define EXT_ENCODER_INDEX_MODEL_AUTO   0x00000002

unsigned int siemensEncoderIndexModel(void)
{
  return EXT_ENCODER_INDEX_MODEL_MANUAL;
}


/*
  siemensEncoderSetIndexModel(unsigned int model)

  For boards that support multiple index models, select which one
  is to be used. Returns 0 if OK, -1 if model can't be supported.
  */
int siemensEncoderSetIndexModel(unsigned int model)
{
  return -1;
}


/*
  siemensEncoderNum()

  returns number of encoders in system.
  */
int siemensEncoderNum(void)
{
  return 8;
}

/*
  siemensEncoderRead(int encoder, double * counts)

  Stores the encoder's counts in counts arg. Returns 0 if
  OK or -1 if encoder is out of range. encoder is in range
  0 .. max encoder - 1.
  */
int siemensEncoderRead(int encoder, double * counts)
{
  return siemensEncoderReadAll(EMCMOT_MAX_AXIS, counts);
}

/*
  siemensEncoderReadAll(int max, double * counts)

  Stores the range of encoders' counts in counts array. Returns 0 if
  OK or -1 if the max is greater than number of encoders. max is
  number of encoders, so for encoders 0..3 max would be 4.
  */
int siemensEncoderReadAll(int max, double * counts)
{
  int t;
  int status3 = ENC_STAT03;
  int status7 = ENC_STAT47;
  char overflow;
  char direction;
  LongWord Encdata;
  if ((max > 7) || (max < 0))
    {
      return -1;
    }
    overflow = (status7 << 4) | (status3 & 0x0f);
    direction = (status7 & 0x0f) | ((status3 & 0xf0) >> 4);
  for (t = 0; t < max; t++)
    {
   Encdata.Long = counts[t];
    if (status3 != ENC_STAT03)
    {
  if ((overflow >> t) & 0x01)
    {
      if ((direction >> t) & 0x01)
        {
          Encdata.Word[1] = Encdata.Word[1] + 1;
        }
      else
        {
          Encdata.Word[1] = Encdata.Word[1] - 1;
	  ;
        }
     }
     }
    Encdata.Word[0] = ENCODER(t);
    counts[t] = Encdata.Long;
  }
  return 0;
}

/*
  siemensEncoderResetIndex(int encoder)

  Resets index latching for the indicated axis. Returns 0 if OK or -1
  if the index is out of range. This applies to both
  EXT_ENCODER_INDEX_MODEL_MANUAL and EXT_ENCODER_INDEX_MODEL_AUTO.
  For the first, it resets the latch flag. For the second, it enables
  zeroing on the next index. encoder is range 0..max encoder - 1.
  */
int siemensEncoderResetIndex(int encoder)
{
  return 0;
}

/*
  siemensEncoderReadLatch(int encoder, int * flag)

  For EXT_ENCODER_INDEX_MODEL_MANUAL, stores 1 if index has latched
  the flag, 0 if not. For EXT_ENCODER_INDEX_MODEL_AUTO, stores 1 if
  the encoder has been zeroed. Returns 0 if OK, -1 if not valid.
  */
int siemensEncoderReadLatch(int encoder, int * flag)
{
  char Byte;
  Byte=(ENC_STAT03 >> 8) & 0x0f;
  Byte=Byte + ((ENC_STAT47 >> 4) & 0xf0);
  *flag=(Byte >> encoder) & 0x01;
  return 0;
}

/*
  siemensEncoderReadLevel(int encoder, int * index)

  For EXT_ENCODER_INDEX_MODEL_MANUAL, stores 1 if encoder is on
  the index pulse right now, 0 if not. Useful for slow polling
  to get more accuracy since the manual model has latency.

  Not valid for EXT_ENCODER_INDEX_MODEL_AUTO. Returns 0 if OK,
  -1 if not valid.
  */
int siemensEncoderReadLevel(int encoder, int * flag)
{
  siemensEncoderReadLatch(encoder, flag);
  return 0;
}

/* Limit switch functions */

/*
  siemensMaxLimitSwitchRead(int axis, int * flag)

  sets *flag to 0 if the limit switch is not tripped, i.e., everything
  is fine, 1 if the limit switch is tripped, i.e., the axis went
  too far in the associated direction.

  Maximum is defined as the direction in which encoder values increase,
  minimum is the other direction.

  Returns 0 if OK, -1 if not valid (axis is out of range).
  */
int siemensMaxLimitSwitchRead(int axis, int * flag)
{
  if (axis > 2 || axis < 0 )
    {
      return -1;
    }
  *flag = siemensDioRead(MAX_LIMIT_BASE_INDEX + axis, flag);
  return 0;
}

int siemensMinLimitSwitchRead(int axis, int * flag)
{
  if (axis > 2 || axis < 0 )
    {
      return -1;
    }
  *flag = siemensDioRead(MIN_LIMIT_BASE_INDEX + axis, flag);
  return 0;
}

/*
  siemensHomeSwitchRead(int axis, int * flag)

  sets *flag to 0 if the home switch is not tripped, i.e., everything
  is fine, 1 if the home switch is tripped.

  Returns 0 if OK, -1 if not valid (axis is out of range).
  */
int siemensHomeSwitchRead(int axis, int * flag)
{
  if (axis > 2 || axis < 0 )
    {
      return -1;
    }
  *flag = siemensDioRead(HOME_BASE_INDEX + axis, flag);
  return 0;
}

/* Amp functions */

/*
   siemensAmpEnable(int axis, int enable)

   enables or disables amplifier for indicated axis; enable flag is
   1 to enable, 0 to disable

   Returns 0 if OK, -1 if not valid (axis is out of range)
   */
int siemensAmpEnable(int axis, int enable)
{
  if (axis > 2 || axis < 0 )
    {
      return -1;
    }
  return siemensDioWrite(axis, enable);
}

/*
  siemensAmpFault(int axis, int * fault)

  Copies into 'fault' the fault state of the amplifier. 1 is faulted,
  0 is OK.

  Returns 0 if OK, -1 if not valid (axis out of range)
  */
int siemensAmpFault(int axis, int * fault)
{
  if (axis > 2 || axis < 0 )
    {
      return -1;
    }
  *fault = siemensDioRead(AMP_FAULT_BASE_INDEX + axis, fault);
  return 0;
}

/*
  Digital I/O model

  If you need to call board-specific code to set up the digital I/O
  registers, do it in siemensDioInit().

  "index" begins at 0.

  Code for each implementation should shift index up into appropriate
  register so that it matches initialization and R/W setup in siemensInit().

  Returns 0 if OK, -1 if error (invalid index).
  */

/* returns the max input index, output index; max bytes, shorts, and words */
int siemensDioMaxInputs(void)       /* index < this for siemensDioRead() */
{
  return 64;
}

int siemensDioMaxOutputs(void)      /* index < this for siemensDioWrite(),Check() */
{
  return 64;
}

/* reads value of digital input at index, stores in value */
int siemensDioRead(int index, int *value)
{
  unsigned short int mask;
  mask = 0x01 << (index % 16);
  if (index >= 0 || index < 16)
    {
      *value = (DIG_RD_BK0 & mask ? 1 : 0);
      return 0;
    }
  if (index >= 16 || index < 32)
    {
      *value = (DIG_RD_BK1 & mask ? 1 : 0);
      return 0;
    }
  if (index >= 32 || index < 48)
    {
      *value = (DIG_RD_BK2 & mask ? 1 : 0);
      return 0;
    }
  if (index >= 48 || index < 64)
    {
      *value = (DIG_RD_BK3 & mask ? 1 : 0);
      return 0;
    }
/* Bad index value */
  return -1;
}

/* writes value (non-zero means 1, 0 is 0) at digital out at index */
int siemensDioWrite(int index, int value)
{
  unsigned short int data;
  int mask;
  mask = (index % 16);
  if (index >= 0 || index < 16)
    {
/* Read the port and OR with the mask before writing it back */
      data = DIG_RD_BK0;
      if (value == 0)
        {
          DIG_WR_BK0(data &= ~(1 << mask));
	}
       else
        {
	  DIG_WR_BK0(data |= (1 << mask));
	}
      return 0;
    }
  if (index >= 16 || index < 32)
    {
      data = DIG_RD_BK1;
      if (value == 0)
        {
          DIG_WR_BK1(data &= ~(1 << mask));
	}
       else
        {
	  DIG_WR_BK1(data |= (1 << mask));
	}
      return 0;
    }
  if (index >= 32 || index < 48)
    {
      data = DIG_RD_BK2;
      if (value == 0)
        {
          DIG_WR_BK2(data &= ~(1 << mask));
	}
       else
        {
	  DIG_WR_BK2(data |= (1 << mask));
	}
      return 0;
    }
  if (index >= 48 || index < 64)
    {
      data = DIG_RD_BK3;
      if (value == 0)
        {
          DIG_WR_BK3(data &= ~(1 << mask));
	}
       else
        {
	  DIG_WR_BK3(data |= (1 << mask));
	}
      return 0;
    }
/* Bad index value */
  return -1;
}

/* reads value of digital OUT at index, stores in value. Useful
   for checking values of previous writes. Returns 0 if OK, -1 if
   bad index or can't read if they're write-only. */
int siemensDioCheck(int index, int *value)
{
  return siemensDioRead(index, value);
}

/* byte, short, and word reads, writes. Index starts at 0, indexes up
   through bytes, short ints, and ints */
int siemensDioByteRead(int index, unsigned char *byte)
{
  switch(index)
    {
      case 0:
        *byte = DIG_RD_BK0 & 0x00ff;
	break;
      case 1:
        *byte = (DIG_RD_BK0 >> 8) & 0x00ff;
	break;
      case 2:
        *byte = DIG_RD_BK1 & 0x00ff;
	break;
      case 3:
        *byte = (DIG_RD_BK1 >> 8) & 0x00ff;
	break;
      case 4:
        *byte = DIG_RD_BK2 & 0x00ff;
	break;
      case 5:
        *byte = (DIG_RD_BK2 >> 8) & 0x00ff;
	break;
      case 6:
        *byte = DIG_RD_BK3 & 0x00ff;
	break;
      case 7:
        *byte = (DIG_RD_BK3 >> 8) & 0x00ff;
	break;
    }
  return 0;
}
int siemensDioShortRead(int index, unsigned short *sh)
{
  switch(index)
    {
      case 0:
        *sh = DIG_RD_BK0;
	break;
      case 1:
        *sh = DIG_RD_BK1;
	break;
      case 2:
        *sh = DIG_RD_BK2;
	break;
      case 3:
        *sh = DIG_RD_BK3;
    }
  return 0;
}
int siemensDioWordRead(int index, unsigned int *word)
{
  switch(index)
    {
      case 0:
        *word = DIG_RD_BK0;
	*word += DIG_RD_BK1 << 16;
	break;
      case 1:
        *word = DIG_RD_BK2;
	*word += DIG_RD_BK3 << 16;
	break;
    }
  return 0;
}
int siemensDioByteWrite(int index, unsigned char byte)
{
  switch(index)
    {
      case 0:
        DIG_WR_BK0((DIG_RD_BK0 & 0xff00) | byte);
	break;
      case 1:
        DIG_WR_BK0((DIG_RD_BK0 & 0x00ff) | (byte << 8));
	break;
      case 2:
        DIG_WR_BK1((DIG_RD_BK1 & 0xff00) | byte);
	break;
      case 3:
        DIG_WR_BK1((DIG_RD_BK1 & 0x00ff) | (byte << 8));
	break;
      case 4:
        DIG_WR_BK2((DIG_RD_BK2 & 0xff00) | byte);
	break;
      case 5:
        DIG_WR_BK2((DIG_RD_BK2 & 0x00ff) | (byte << 8));
	break;
      case 6:
        DIG_WR_BK3((DIG_RD_BK3 & 0xff00) | byte);
	break;
      case 7:
        DIG_WR_BK3((DIG_RD_BK3 & 0x00ff) | (byte << 8));
	break;
    }
  return 0;
}
int siemensDioShortWrite(int index, unsigned short sh)
{
  switch(index)
    {
      case 0:
        DIG_WR_BK0(sh);
	break;
      case 1:
        DIG_WR_BK1(sh);
	break;
      case 2:
        DIG_WR_BK2(sh);
	break;
      case 3:
        DIG_WR_BK3(sh);
    }
  return 0;
}
int siemensDioWordWrite(int index, unsigned int word)
{
  switch(index)
    {
      case 0:
        DIG_WR_BK0(word & 0xffff);
	DIG_WR_BK1((word >> 16) & 0xffff);
	break;
      case 1:
        DIG_WR_BK2(word & 0xffff);
	DIG_WR_BK3((word >> 16) & 0xffff);
	break;
    }
  return 0;
}
int siemensDioByteCheck(int index, unsigned char *byte)
{
  return 0;
}
int siemensDioShortCheck(int index, unsigned short *sh){
  return 0;
}
int siemensDioWordCheck(int index, unsigned int *word)
{
  return 0;
}

/* writes a byte of IO synched with motion start/end */
int siemensMotDout(unsigned char byte)
{
  return 0;
}

/* Internal functions */

void RawDacOut(int dac, double volts)
{
  unsigned short int RawVolts;		/* Limit resolution to 16 bits */
  unsigned short int tmpflush;
  unsigned long PAA0=0x60;
		
/* Convert float volts to unsigned hex
    -10V = 0x0000
      0V = 0x0800
    +10V = 0x0fff */

  /* -10 V = 0x8000
   *   0 V = 0x0000
   * +10 V = 0x7fff
   */
  
  if (volts > 10)		/* Limit the range for DAC output */
    {
      volts = 10;
    }
  if (volts < -10)
    {
      volts = -10;
     }
  RawVolts = ((volts/10) * 0x7fff); /* and convert to Hex value */
  
  switch(dac)
    {
      case 0:
	// DAC_CHAN0(RawVolts);

	tmpflush = readw(card_address + PAA0);
	writew(RawVolts, card_address + PAA0);
//	writew(0x7fff, card_address + PAA0);
	tmpflush = readw(card_address + PAA0);
	
	break;
      case 1:
        //DAC_CHAN1(RawVolts);
	break;
      case 2:
        //DAC_CHAN2(RawVolts);
	break;
    }
 return;
}

